% file: example5.m	G. Clifford and G. Moody	27 November 2003

fprintf('example5.m - demo of annotation processing using WFDB_tools\n')

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This script is part of a brief 'hands on' tutorial for the WFDB_tools
% package.  Before running this script, you will need to have correctly
% installed the WFDB software package and added the Matlab WFDB_tools
% path at the Matlab prompt.
%
% Please report any problems with this script to Gari Clifford:
%	gari AT physionet DOT org
% Please report any problems with WFDB_tools for Matlab to Jonas Carlson:
%       jonas DOT carlson AT kard DOT lu DOT se
% Please report any problems with the WFDB software package to George Moody:
%       george AT physionet DOT org

% Set pauseflag to 0 before running this script to disable pauses.
if(~exist('pauseflag')) pauseflag=1; end

% Check that the WFDB_tools functions are installed and accessible.
if(exist('WFDB_isigopen')==0)
  fprintf('Please add WFDB_tools to the PATH ...');
  if(isunix)
     error(' e.g.,\n     addpath ~/matlab/WFDB_tools/\n');
  elseif(ispc)
     error(' e.g.,\n     addpath c:\\matlab\\WFDB_tools\n');
  else
     error('... ');
  end
end

% Check that the WFDB software package is installed and accessible.
if system('rdann -r mitdb/201 -a atr -c 1')
  fprintf('The WFDB software package does not seem to be installed.\n');
  error('Install it and add the directory containing "rdann" to your PATH.\n');
end

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

% End of standard setup.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Choose an input record.
record = 'mitdb/201';

% Open the signals.
fprintf('We%sll begin by opening the input signals, using\n',setstr(39));
fprintf('   hea = WFDB_isigopen(record);\n');
fprintf('Information about the signals is loaded into "hea":\n');
hea = WFDB_isigopen(record)
hea.nsamp;
sampfreq = WFDB_getifreq;

% Open the reference annotation (.atr) file for the input record.
A = WFDB_Anninfo(1);
A.name='atr';
WFDB_annopen(record,A);

% Read the first 75 annotations
Ann = WFDB_getann(0,75);
% and copy their time-of-occurrence and type attributes into two vectors
for(i=1:75);
  BeatIndex(i) = Ann(i).time;
  label(i) = Ann(i).anntyp;
end

NORMAL=WFDB_strann('N');

% Now find the normal beats (those for which label(i) is NORMAL) ...
N = find(label==NORMAL);
% ... and the abnormals (note that this will also find non-beat events, such
% as rhythm and signal quality annotations).
A = find(label~=NORMAL);

% now get the data ...
TimeOfInterest = Ann(75).time/sampfreq; % elapsed time in seconds
data = WFDB_getvec(length(hea),sampfreq*TimeOfInterest);

% now plot the data
t = [1/sampfreq:1/sampfreq:TimeOfInterest];
figure;

subplot(2,1,1) % signal 0
plot(t,data(:,1),'b')
hold on
plot(t(BeatIndex(N)),data(BeatIndex(N),1),'g+') % plot normals
plot(t(BeatIndex(A)),data(BeatIndex(A),1),'r+') % plot abnormal 

subplot(2,1,2) % signal 1
plot(t,data(:,2),'b')
hold on
plot(t(BeatIndex(N)),data(BeatIndex(N),2),'g+') % plot normals
plot(t(BeatIndex(A)),data(BeatIndex(A),2),'r+') % plot abnormal 

fprintf('Notice that there are 73 normal beats (green crosses) and 2 other\n');
fprintf('events (red crosses).  Note also that the fiducial markers are\n');
fprintf('not always on the exact R-peak, because they have been placed by\n');
fprintf('human experts and then moved to the QRS centroids.\n');
fprintf('\nPress any key to zoom in on them.\n\n')

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

subplot(2,1,1); axis([0 1.6 800 1300])
subplot(2,1,2); axis([46 50 800 1300])

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

% Ann(1).anntyp;
fprintf('The first label in the top plot has a value of %i ... \n',Ann(1).anntyp);
fprintf('which means "%s".\n\n',WFDB_anndesc(Ann(1).anntyp));
fprintf('The label for the sixth beat in the lower plot has a value %i\n',Ann(74).anntyp);

fprintf('meaning "%s",\n\n',WFDB_anndesc(Ann(74).anntyp));
fprintf('Notice that the aberrated atrial premature beat occurs a little\n');
fprintf('early with respect to the other beats.\n')
fprintf('Now we will look at beat-to-beat intervals in this data segment.\n')

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

% ignore first marker point and find Delta RR
DeltaRR = diff(t(BeatIndex(2:75)));
tDeltaRR = t(BeatIndex(3:75));

figure;
subplot(2,1,1);plot(t,data(:,1))
hold on;
plot(t(BeatIndex(N)),data(BeatIndex(N),1),'g+') % plot normals
plot(t(BeatIndex(A)),data(BeatIndex(A),1),'r+') % plot others

subplot(2,1,2);plot(tDeltaRR,DeltaRR,'g+--');hold on
plot(tDeltaRR(A(2)-2),DeltaRR(A(2)-2),'r+')

fprintf('Notice the significantly shortened RR interval (marked with a red\n');
fprintf('cross in the lower plot) associated with the premature beat.\n');

fprintf('\n\nWe finish by closing all files with WFDB_wfdbquit\n');
WFDB_wfdbquit
fprintf('\nWFDB_wfdbquit successful, end of example5.m\n');
