% file: example4.m	G. Clifford and G. Moody	27 November 2003

fprintf('example4.m - demonstration of writing signals using WFDB_tools\n')

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This script is part of a brief 'hands on' tutorial for the WFDB_tools
% package.  Before running this script, you will need to have correctly
% installed the WFDB software package and added the Matlab WFDB_tools
% path at the Matlab prompt.
%
% Please report any problems with this script to Gari Clifford:
%	gari AT physionet DOT org
% Please report any problems with WFDB_tools for Matlab to Jonas Carlson:
%       jonas DOT carlson AT kard DOT lu DOT se
% Please report any problems with the WFDB software package to George Moody:
%       george AT physionet DOT org

% Set pauseflag to 0 before running this script to disable pauses.
if(~exist('pauseflag')) pauseflag=1; end

% Check that the WFDB_tools functions are installed and accessible.
if(exist('WFDB_isigopen')==0)
  fprintf('Please add WFDB_tools to the PATH ...');
  if(isunix)
     error(' e.g.,\n     addpath ~/matlab/WFDB_tools/\n');
  elseif(ispc)
     error(' e.g.,\n     addpath c:\\matlab\\WFDB_tools\n');
  else
     error('... ');
  end
end

% Check that the WFDB software package is installed and accessible.
if system('rdann -r mitdb/201 -a atr -c 1')
  fprintf('The WFDB software package does not seem to be installed.\n');
  error('Install it and add the directory containing "rdann" to your PATH.\n');
end

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

% End of standard setup.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Writing signals using WFDB_tools is a 4-step process:
%  1. Create and fill in signal info
%  2. Write output signals
%  3. Create header file
%  4. Close the open signal and header files

fprintf('First, we create signal information structures for 3 signals:\n');
fprintf('   S = WFDB_Siginfo(3)\n')
S = WFDB_Siginfo(3)

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('Next, we fill these structures with appropriate values ... \n')
fprintf('(all of them have appropriate default values to avoid errors, but\n');
fprintf('it is unlikely that the defaults are appropriate for our signals)\n');

fprintf('\nFor example, if signal 0 is synthesized noise, we can write\n');
fprintf('   S(1).desc = %sNoise%s\n',setstr(39),setstr(39));
S(1).desc = 'Noise';

fprintf('\nRemember, the first signal is signal 0, and its attributes are\n');
fprintf('given by S(1).  We can similarly set any of the other fields of S\n');
fprintf('for which the default values are not correct:\n');
S(2).desc = 'Cosine - 1 Hz'
S(3).desc = 'Sine - 0.25 Hz'

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('When we%sve finished filling out all of the desired fields,\n',setstr(39));
fprintf('we can create an empty signal file by:\n');
fprintf('   WFDB_osigfopen(S)\n\n')
WFDB_osigfopen(S)

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('We specify the sampling frequency (in this example, 1000Hz) by:\n');
fprintf('   WFDB_setsampfreq(1000))\n');
WFDB_setsampfreq(1000)

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('We can also set the basetime (start time) of the recording, by:\n');
fprintf('   WFDB_setbasetime(%s02:19:00 02/09/1999%s)\n',setstr(39),setstr(39));
WFDB_setbasetime('02:19:00 02/09/1999')
fprintf('Note that dates are always in DD/MM/YYYY format, so the time in\n');
fprintf('this example is 19 minutes after 2 AM on 2 September 1999.\n');

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('Now we write the samples of the signals, which must be stored\n');
fprintf('column-wise in a matrix.\n\n');

fprintf('We%sll create 10 seconds of data at 1000 Hz (10000 samples of\n',setstr(39));
fprintf('each of the three signals).  The first column of the data matrix\n');
fprintf('will contain samples for signal 0 (noise generated using "rand"),\n');
fprintf('and columns 2 and 3 will contain samples of signals 1 and 2,\n');
fprintf('(1 Hz cosine and 0.25 Hz sine waves, respectively).\n');

% Fill the data matrix.
i=[1:1:10000];
data1    = rand(10000,1)*500;
data2(i) = 100*cos(2*pi*i/1000);
data3(i) = 250*sin(2*pi*i/4000);
data     = [data1 data2' data3'];

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('The contents of the data matrix can be written to the signal file\n');
fprintf('using:\n');
fprintf('   WFDB_putvec(data);\n')
WFDB_putvec(data);

fprintf('In order to be able to read the signal file later, we need to\n');
fprintf('create a header (.hea) file that describes it, using\n');
fprintf('   WFDB_newheader(%stest1%s);\n',setstr(39),setstr(39));
WFDB_newheader('test1');
fprintf('The argument we give to WFDB_newheader is the record name for the\n');
fprintf('newly created record (in this example, "test1").\n');

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('\n\nWe finish by closing all files with WFDB_wfdbquit:\n');
WFDB_wfdbquit              % close all open files 
fprintf('\nWFDB_wfdbquit successful\n\n')
fprintf('Note: we must create the header file before using WFDB_wfdbquit!\n');

if(pauseflag==1)
  fprintf('\nPress any key to view the results using WAVE\n');
  fprintf('if it is available: ');
  pause;
end;

if system('wave -r test1')
  fprintf('WAVE does not appear to be installed.  Here are\n');
  fprintf('the first few samples, as read using "rdsamp":\n\n');
  if system('rdsamp -r test1 -t s10')
    error('rdsamp was not successful\n');
  end;
end;

fprintf('End of example4.m\n');
